create or replace PACKAGE MIGRATION_UTILS 
AS
    PROCEDURE COPY_DATA(TABLE_NAME IN VARCHAR2, CREDENTIAL_NAME IN VARCHAR2, FILE_URI_LIST IN CLOB, SCHEMA_NAME IN VARCHAR2,          
               FORMAT IN CLOB, DTTYPE IN VARCHAR2);
END MIGRATION_UTILS;
/
create or replace PACKAGE BODY MIGRATION_UTILS 
AS
    TYPE VARCHAR2_ARRAY IS TABLE OF VARCHAR2(100);
    DT_FORMATS          VARCHAR2_ARRAY; 
    DT_DAY              VARCHAR2_ARRAY; -- Datetime formats starting with Day
    DT_MONTH            VARCHAR2_ARRAY; -- Datetime formats starting with Month
    DT_YEAR             VARCHAR2_ARRAY; -- Datetime formats starting with Year
    DT_TIME             VARCHAR2_ARRAY; -- Datetime formats just having Hour, Minute, Second and Fractional seconds

PROCEDURE COPY_DATA(TABLE_NAME IN VARCHAR2, CREDENTIAL_NAME IN VARCHAR2, FILE_URI_LIST IN CLOB, SCHEMA_NAME IN VARCHAR2,          
                FORMAT IN CLOB, DTTYPE IN VARCHAR2) AS
v_json JSON_OBJECT_T;
BEGIN
    FOR i in DT_FORMATS.FIRST .. DT_FORMATS.LAST
    LOOP
      BEGIN
        v_json := JSON_OBJECT_T.parse(FORMAT);
        IF UPPER(DTTYPE) = 'DATE' THEN
          v_json.put('dateformat', DT_FORMATS(i));
        ELSIF UPPER(DTTYPE) = 'TIMESTAMP' THEN
          v_json.put('timestampformat', DT_FORMATS(i));
        ELSIF UPPER(DTTYPE) = 'TIMESTAMP WITH TIME ZONE' THEN
          v_json.put('timestamptzformat', DT_FORMATS(i));
        ELSIF UPPER(DTTYPE) = 'TIMESTAMP WITH LOCAL TIME ZONE' THEN
          v_json.put('timestampltzformat', DT_FORMATS(i));
        END IF;
        
        IF v_json.to_string IS JSON THEN 
            DBMS_OUTPUT.PUT_LINE(v_json.to_string);
            DBMS_CLOUD.COPY_DATA(
              schema_name => SCHEMA_NAME, 
              table_name => TABLE_NAME, 
              credential_name => CREDENTIAL_NAME, 
              file_uri_list => FILE_URI_LIST,
              format => v_json.to_string
            );
       END IF;     
      EXIT;
      EXCEPTION
       WHEN OTHERS THEN
        DBMS_OUTPUT.PUT_LINE(DBMS_UTILITY.FORMAT_ERROR_STACK());
        NULL;
      END;
    END LOOP;
END COPY_DATA;

BEGIN    
	-- Datetime formats/styles starting with DAY only --
	DT_DAY := VARCHAR2_ARRAY(
    ----- Format/Style ------       ------ Standard ------ 
	'fxfmdd/mm/yy',                    --British/French
	'fxfmdd/mm/yyyy',                  --British/French(with Century)
	'fxfmdd.mm.yy',                    --German(without Century)
	'fxfmdd.mm.yyyy',                  --German(with Century)
	'fxfmdd-mm-yy',                    --Italian(without Century)
	'fxfmdd-mm-yyyy',                  --Italian(with Century)
	'fxfmdd mon yy',                   --(without Century)
	'fxfmdd mon yyyy',                 --(with Century)
	'dd mon yyyy hh24:mi:ssxff3',      --Europe default+milliseconds
	'fxdd mon yyyy hh12:mi:ss:ff3AM',  --Hijri calendar system 
	'fxdd/mm/yy hh12:mi:ss:ff3AM',     --Hijri calendar system 
	
	-------- Some more DateTime formats that were addressed before --------
	'fmdd yyyy MONTH',
	
	----------------- Additional DateTime formats ------------------------- 
	'fmdd Month',
	'fmdd Month yy',
	'fmdd Month yyyy',
	'fxddmmyy',
	'fxddmmyyyy',
	'fxdd-Mon-yy',
	'fxdd-MON-yy',
	'fxdd-Mon-yyyy',
	'fxdd-MON-yyyy'
	);
	
	-- Datetime formats/styles starting with MONTH only --
	DT_MONTH := VARCHAR2_ARRAY(
	----- Format/Style ------       ------ Standard ------ 
	'fxfmmm/dd/yyyy hh24:mi:ss',       --SQLSERVER 101
	'fxfmmm-dd-yyyy hh24:mi:ss',
	'fxfmmm.dd.yyyy hh24:mi:ss',
	'fmmon dd yyyy hh:miAM',           --Default   
	'fxfmmm/dd/yy',                    --US(without Century)
	'fxfmmm-dd-yy',
	'fxfmmm.dd.yy', 
	'fxfmmm/dd/yyyy',                  --US(with Century)
	'fxfmmm-dd-yyyy',
	'fxfmmm.dd.yyyy',
	'fxfmMon dd, yy',                  --(without Century)
	'fxfmMon dd, yyyy',                --(with Century)
	'mon dd yyyy hh12:mi:ssxff3am',    --Default+milliseconds
	
	-------- Some more DateTime formats that were addressed before -------- 
	'fmMONTH, yyyy',
	'MON yyyy',
	'fmMONTH dd, yyyy',
	----------------- Additional DateTime formats ------------------------- 
	'mm/yy',
	'mm/yyyy',
	'fmMonth dd, yyyy',             
	'fmMonth dd',
	'mm-yy',
	'mm-yyyy',
	'fxmmddyy',
	'fxmmddyyyy',
	'Mon-yy',
	'Mon-yyyy',
	'MON-yy',
	'MON-yyyy'
	);
	
	-- Datetime formats/styles starting with YEAR only --
	DT_YEAR := VARCHAR2_ARRAY(
    'auto',                             -- uses DBMS_CLOUD.COPY_DATA(..) defaults dt formats 
	----- Format/Style ------       ------ Standard ------ 
	'fxfmyy.mm.dd',                    --ANSI(without Century)
	'fxfmyyyy.mm.dd',                  --ANSI(with Century)
	'fxfmyy/mm/dd',                    --Japan(without Century)
	'fxfmyyyy/mm/dd',                  --Japan(with Century)
	'yymmdd',                          --ISO(without Century)
	'yyyymmdd',                        --ISO(with Century)
	'fxyyyy-mm-dd hh24:mi:ss',         --ODBC canonical
	'fxyyyy-mm-dd hh24:mi:ssxff3',     --ODBC canonical (with milliseconds)
	'fxyyyy-mm-dd"T"hh12:mi:ssxff3',   --ISO8601 (no spaces)
	'fxyyyy-mm-dd hh12:mi:ssxff3',     --ISO8601 (with space)
	'fxyyyy-mm-dd hh12:mi:ssxff3tzr',  --ISO8601 with time zone Z
	
	-------- Some more DateTime formats that were addressed before -------- 
	'fxyyyymmdd hh24:mi:ss',
	'fxyyyy-mm-dd',                    --ISO8601 Date only
	'fxyyyy-mm-dd hh12:mi:ss',
	'yyyy MON',
	
	----------------- Additional DateTime formats ------------------------- 
	'yy/mm',
	'yyyy/mm',
	'yy-mm',
	'yyyy-mm'
	);
	
	-- TIME formats/styles --
	DT_TIME := VARCHAR2_ARRAY(
	----- Format/Style ------       ------ Standard ------ 
	'hh:mi:ss',
	'hh24:mi:ssxff3',
	
	-------- Some more Time formats that were addressed before ------------ 
	'hh12 AM',                         --Time using Hour
	'hh12:mi:ss AM'                    --Time
	);
	DT_FORMATS := DT_YEAR MULTISET UNION DT_MONTH MULTISET UNION DT_DAY MULTISET UNION DT_TIME;

END MIGRATION_UTILS;
/